/* --COPYRIGHT--,BSD
 * Copyright (c) 2011, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
/*****************************************************************************************/
// Description:  The MSP430 detects when both sides of the enclosure are 
// touched, representing someone holding or 'gripping' the enclosure.
// Once two successive measurements detect a grip an LED indication is 
// presented and then the LEDs represent which button, if any, is pressed.
//
//                MSP430G2553
//             -----------------
//         /|\|              XIN|-
//          | |                 |
//          --|RST          XOUT|-
//          --|TEST             |
//            |             P2.0|--<Input, "Left Grip"
//            |             P2.5|--<Input, "Right Grip"
//            |             P1.5|--<Input, "BUT1"
//            |             P1.6|--<Input, "BUT2"
//            |                 |
//            |             P1.4|-->Output, "LED1"
//            |             P1.7|-->Output, "LED2"
//             -----------------
//
// GripDetection_BoosterPack.c
//   1) Use the newest Cap Touch Library in SLAA489
//   2) Support ELEMENT_CHARACTERIZATION_MODE
//   3) Support GUI_DISPLAY, can communicate with TouchPro_1_01_00_00 GUI using 8MHz
//      Clock and 9600 Baudrate
//
// The Original version is created by Tyler B  
// The latest version is created by Holly G 10/06/2013
//*****************************************************************************************/


#include "CTS_Layer.h"


//-------------------------------------------------------------------------------------
// ELEMENT_CHARACTERIZATION_MODE is used for parameters adjustment, like measGateSource
// accumulationCycles,threshold defined in structure.c file. If enabled, set breakpoint
// to __no_operation() to observe corresponding registers like deltaCnt and deltaCnt_1
// to set different parameters. This mode is recommended to be used at the very beginning
// of the design.
//-------------------------------------------------------------------------------------
//#define ELEMENT_CHARACTERIZATION_MODE

//-------------------------------------------------------------------------------------
// GUI_DISPLAY is used for communicating with the TouchPro_1_01_00_00 using UART, the
// Baudrate is 9600. Users can easily observe the cap touch performance in the GUI
//-------------------------------------------------------------------------------------
//#define GUI_DISPLAY

//-------------------------------------------------------------------------------------
// Define some delays used as the parameters called by the function
// void sleep(uint16_t time). The clock source is ~11KHz VLO clock
// LED_SEQUENCE: Time Between LED blinks when grip is first detected
//               1400 * (1/11) = ~127ms
// ACTIVE_DELAY: Time Between measurements in active mode, when grip is detected, it
//               means that every ~100ms, the TIMERA interrupt will wake the micro up
//               from LPM3 to see if grip is still existed and if any button is pressed
//               1100 * (1/11) = ~100ms
// IDLE_DELAY  : Time Between measurements in idle mode, when no grip is detected, it
//               means that the TIMERA interrupt will wake the micro up from LPM3 every
//               ~999ms to detect if there's any grip gestures presented.
//               10989 * (1/11) = ~999ms
//-------------------------------------------------------------------------------------
#define LED_SEQUENCE             1400
#define ACTIVE_DELAY             1100
#define IDLE_DELAY               10989

//-------------------------------------------------------------------------------------
// Define the number of Grip sensors and Buttons in the designed, if these settings are
// changed, TOTAL_NUMBER_OF_ELEMENTS as well as MAXIMUM_NUMBER_OF_ELEMENTS_PER_SENSOR in
// structure.h should also be changed accordingly
//-------------------------------------------------------------------------------------
#define NUM_GRIP_SEN              2
#define NUM_BUT_SEN               2
#define SENSOR_NUM                4


//-------------------------------------------------------------------------------------
// Definition some functions for Uart communication to GUI
//-------------------------------------------------------------------------------------
void Data_frame_send(void); 		// Cap Touch Display Data Logger
void UART_send_byte(uint8_t data2send);
void USCI_config_UART(void);
void TimerA_UART_print(uint8_t *string);

#define DataHead 0x55AA     //Head of the UART GUI communication Protocal

uint16_t deltaCnt[NUM_GRIP_SEN];
uint16_t deltaCnt_1[NUM_BUT_SEN];
uint16_t count[SENSOR_NUM];

uint8_t channelIndex;  // The channels need to be displayed in the GUI, also the sensor number

//-------------------------------------------------------------------------------------
// Initilize Clock, set SMCLK = MCLK = 8Mhz, VLO as ACLK when enter LPM3
//-------------------------------------------------------------------------------------
void Clock_Initial()
{
    BCSCTL1 = CALBC1_8MHZ;
    DCOCTL = CALDCO_8MHZ;
    BCSCTL2 |= DIVS_0;                    // SMCLK = MCLK = 8Mhz
    BCSCTL3 |= LFXT1S_2;                  // LFXT1 = VLO
}

//-------------------------------------------------------------------------------------
// Initilize All the GPIO ports
//-------------------------------------------------------------------------------------
void Port_Initial()
{
    P1SEL = 0x00;
    P1SEL2 = 0x00;
    P1DIR = 0xFF;
    P1OUT = 0x00;
    P2DIR = 0xFF;
    P2OUT = 0x00;
    P2SEL = 0x00;
}

//------------------------------------------------------------------------------
// Sleep function enters LPM3 using TIMER0.0 sourced by VLO clock. After 'time',
// the interrupt will wake up the micro.
//------------------------------------------------------------------------------
void sleep(uint16_t time)
{
    TA0CCR0 = time;
    TA0CTL = MC_1+TACLR+TASSEL_1; 
    TA0CCTL0 &= ~(CAP+CCIFG);
    TA0CCTL0 |= CCIE;
    __bis_SR_register(LPM3_bits+GIE);
    __no_operation();
}

void main(void)
{   
    WDTCTL = WDTPW + WDTHOLD;             // Stop Watchdog timer
    Clock_Initial();                      // Initialize Clock
    Port_Initial();                       // Initialize all the GPIOs
#ifdef GUI_DISPLAY
	USCI_config_UART();					  // Initialize UART Communication at 9600 with a 8MHz DCO frequency
#endif

    // Initialize Baseline Tracking
    TI_CAPT_Init_Baseline(&Grip);
    TI_CAPT_Update_Baseline(&Grip,2);
    TI_CAPT_Init_Baseline(&Buttons);
    TI_CAPT_Update_Baseline(&Buttons,2);

#ifndef ELEMENT_CHARACTERIZATION_MODE
    uint8_t GRIP = 0x00;
#endif

    while (1)
    {
#ifdef ELEMENT_CHARACTERIZATION_MODE
    TI_CAPT_Custom(&Grip,deltaCnt);
	TI_CAPT_Custom(&Buttons,deltaCnt_1);
	__no_operation(); 					  // Set breakpoint here for parameter adjustment

#ifdef GUI_DISPLAY
	count[0] = deltaCnt[0];          // count[0] & count[1] for Grip sensors
	count[1] = deltaCnt[1];
	count[2] = deltaCnt_1[0];        // count[2] & count[3] for Buttons
	count[3] = deltaCnt_1[1];
	Data_frame_send();				 //Cap Touch Display Data Logger Call
	__no_operation(); 				 // Set breakpoint here for GUI interface debug
#endif

#endif


#ifndef ELEMENT_CHARACTERIZATION_MODE

        // Turn off LEDs before measurements
  	    P1OUT &= ~(BIT4 + BIT7);

  	    // Update the status for Two Grip Detection sensors
        TI_CAPT_Custom(&Grip,deltaCnt);

        if(deltaCnt[0] > Right.threshold && deltaCnt[1] > Left.threshold) // Grip is detected
        {
            if(GRIP)
            {
                if(GRIP == 1)  //If this is the first grip when wake up from LPM3, blink LED three times
                {
                	  P1OUT |= BIT4 + BIT7;
                    sleep(LED_SEQUENCE);
                	  P1OUT &= ~(BIT4 + BIT7);
                    sleep(LED_SEQUENCE);
                	  P1OUT |= BIT4 + BIT7;
                    sleep(LED_SEQUENCE);
                	  P1OUT &= ~(BIT4 + BIT7);
                    GRIP=2;
                }

                // Update the status for Two Buttons
                TI_CAPT_Custom(&Buttons,deltaCnt_1);
                if(deltaCnt_1[0] > BUT1.threshold) // Button is detected, blink corresponding LED to indicate which button is pressed
                {
                    P1OUT |= BIT4;
                }
                else 
                {
                    P1OUT &= ~BIT4;
                }
                if(deltaCnt_1[1] > BUT2.threshold)
                {
                    P1OUT |= BIT7;
                }
                else 
                {  
                    P1OUT &= ~BIT7;
                }
            }
            else
            {
                GRIP=1;  // The first time detect grip event, set grip = 1;
            }

            // Enter LPM3 for ACTIVE_DELAY time, until TIMERA interrupt wake the micro up for another round Grips and Buttons Detection in active mode
            sleep(ACTIVE_DELAY);
        }
        // The Grip is no longer presented, Enter LPM3 for IDLE_DELAY time to save power
        // until TIMERA interrupt wake the micro up for a new Grip Detection
        else 
        {
            GRIP = 0x00;
            sleep(IDLE_DELAY);
        }

#ifdef GUI_DISPLAY
	count[0] = deltaCnt[0];             // count[0] & count[1] for Grip sensors
	count[1] = deltaCnt[1];
	count[2] = deltaCnt_1[0];           // count[2] & count[3] for Buttons
	count[3] = deltaCnt_1[1];
	Data_frame_send();				    //Cap Touch Display Data Logger Call
	__no_operation(); 					// Set breakpoint here for GUI interface debug
#endif
#endif
    }
}

// Uart communication is not useful until GUI_DISPLAY is defined
#ifdef GUI_DISPLAY
//------------------------------------------------------------------------------
// 	USCIA - USCI Module Configuration for UART, TX ONLY!
// 	Note: Requires MSP430G2xx3 (Or other device with USCI)
//------------------------------------------------------------------------------
void USCI_config_UART(void)
{

	P1SEL	|= BIT2;			// Specilize GPIO for UART Communication
	P1SEL2	|= BIT2;

	UCA0CTL1 	|= UCSWRST;   	// Put State Machine in Reset
	UCA0CTL1 	|= UCSSEL_2;  	// Select SMCLK
	UCA0BR0 = 65;               // 8MHz 9600
	UCA0BR1 = 3;                // 8MHz 9600
	UCA0MCTL = UCBRS1;          // Modulation UCBRSx = 2
	UCA0CTL1 &= ~UCSWRST;       // Initialize USCI State Machine

}

//------------------------------------------------------------------------------
// 	USCIA - UART TX Function
// 	Note: Requires MSP430G2xx3 (Or other device with USCI)
// 	Input - unsigned char data2send - byte of data to transmit
//------------------------------------------------------------------------------

void UART_send_byte(uint8_t data2send)
{
	  while (!(IFG2 & UCA0TXIFG));	    // Wait for USCI Buffer to be cleared
	  UCA0TXBUF = data2send;			// Load data into USCI TX Buffer to send
}

//------------------------------------------------------------------------------
// Prints a string over using the Timer_A UART
//------------------------------------------------------------------------------
void TimerA_UART_print(uint8_t *string)
{
    while (*string) {
    	UART_send_byte(*string++);
    }
}

//------------------------------------------------------------------------------
// Function calculates data length and check sum. This is the
// Cap Touch Display tool protocol, communication with GUI in PC side
//------------------------------------------------------------------------------

void Data_frame_send(void) {
	uint8_t check_sum = 0;
	uint8_t i = 0;
	uint16_t temp = 0;
	uint8_t data_length = 0;

	data_length = SENSOR_NUM * 3  + 1;    //# of channels *3 + 1 (checksum).
										  //1 byte for the channel number, 1 byte for high order data byte
										  //and 3rd byte is for low order byte.  The plus 1 is for checksum byte.

	temp = DataHead;
	temp >>= 8; 							//Send high byte
	UART_send_byte((uint8_t) temp);
	check_sum = (uint8_t) temp;

	UART_send_byte((uint8_t) DataHead);
	check_sum += (uint8_t) DataHead;

	UART_send_byte(data_length);
	check_sum += data_length;

		for (i = 0; i < SENSOR_NUM; i++) {
		UART_send_byte(i + 1);
		check_sum += i + 1;

		temp = count[i];
		temp >>= 8; //Send high byte

		UART_send_byte((uint8_t) temp);
		check_sum += (uint8_t) temp;

		UART_send_byte((uint8_t) count[i]);
		check_sum += (uint8_t) count[i];
	}
	UART_send_byte((uint8_t) check_sum);
}
#endif

//------------------------------------------------------------------------------
// 	Timer A0 ISR, exit LPM3 which is setup in sleep() function.
//------------------------------------------------------------------------------
#pragma vector = TIMER0_A0_VECTOR
__interrupt void Timer0_A0 (void)
{
    TA0CTL &= ~(MC_1);
    TA0CCTL0 &= ~(CCIE);
    __bic_SR_register_on_exit(LPM3_bits+GIE);
}
